﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using TMPro;
using UnityEngine;
using UnityEngine.UI;

//MissionTimerController manages the mission timer UI
public class MissionTimerController : MonoBehaviour
{
    //UI Elements
    public TMP_Text TimeText;
    public Image TickFillImage;
    public Animator Animator;

    public bool IsEnabled;

    private float _StartTime;   //The time we started it
    private bool _PlayingWarn = false;  //Are we warning the player with the SFX?
    private Stack<float> _Times = new Stack<float>();   //The times at which to warn the player

    /// <summary>
    /// Adds a time at which to warn the player with the SFX
    /// </summary>
    /// <param name="time">The time to add (ms)</param>
    public void AddTime(float time)
    {
        _Times.Push(time);
    }

    /// <summary>
    /// Starts the timer from a corresponding start time
    /// </summary>
    /// <param name="startTime">The time at which to start from</param>
    public void StartTimer(float startTime)
    {
        ResetTimer();
        _Times = new Stack<float>(_Times.OrderBy(i => i));
        _StartTime = startTime;
        IsEnabled = true;
    }

    /// <summary>
    /// Stops the timer by disabling it
    /// </summary>
    public void StopTimer()
    {
        IsEnabled = false;
    }

    /// <summary>
    /// Resets the timer's properties and UI
    /// </summary>
    public void ResetTimer()
    {
        _StartTime = 0.0f;
        TickFillImage.fillAmount = 1.0f;
        _PlayingWarn = false;
        Animator.Play("Idle");
    }

    /// <summary>
    /// Restarts the timer at the specified start time
    /// </summary>
    /// <param name="startTime">The time at which to start from</param>
    public void RestartTimer(float startTime)
    {
        ResetTimer();
        _StartTime = startTime;
        IsEnabled = true;
    }

    /// <summary>
    /// Updates the timer, based on a reference time for warning the player and animating
    /// </summary>
    /// <param name="referenceTime">The reference time for behaviour</param>
    public void UpdateTimer(float referenceTime)
    {
        if(IsEnabled)
        {
            if (referenceTime <= _Times.Peek())
            {
                //Our reference time is now below or at a new warn time, let's play the SFX then pop it off
                AudioManager.Instance.PlayFile(AudioManager.Instance.LoadedFiles["CountdownSFX"]);
                _Times.Pop();
            }

            //Set up the UI
            TickFillImage.fillAmount = referenceTime / _StartTime;
            TimeSpan timeSpan = TimeSpan.FromMilliseconds(referenceTime);
            TimeText.text = string.Format("{0}:{1:00}", (int)timeSpan.TotalMinutes, timeSpan.Seconds);


            if(!_PlayingWarn && referenceTime <= 10000.0f)
            {
                //10s or less to go, let's warn if we're not already
                Animator.Play("Warn");
                _PlayingWarn = true;
            }

            if(referenceTime <= 0.0f)
            {
                //Time up, idle back
                Animator.Play("Idle");
            }
        }
    }
}
